import psutil
import sys
import requests

def is_valid_ip(ip):
    return not (ip.startswith('169.254.') or ip == '0.0.0.0' or ip in ['::', '::1'])

def get_ip_info(ip):
    try:
        response = requests.get(f"http://ip-api.com/json/{ip}", timeout=5)
        data = response.json()
        if data["status"] == "success":
            return f"{data['country']} - {data['regionName']}, {data['city']} | ISP: {data['isp']}"
        else:
            return "IP info unavailable"
    except Exception:
        return "IP lookup failed"

def show_connections(port_filters=None):
    seen = set()
    connections = [
        conn for conn in psutil.net_connections(kind='inet')
        if conn.status == psutil.CONN_ESTABLISHED and conn.raddr
    ]

    enriched = []
    for conn in connections:
        try:
            proc = psutil.Process(conn.pid) if conn.pid else None
            start_time = proc.create_time() if proc else 0
        except Exception:
            start_time = 0
        enriched.append((start_time, conn))

    enriched.sort(key=lambda x: (x[0], x[1].fd if x[1].fd is not None else -1))

    for _, conn in enriched:
        local_ip, local_port = conn.laddr
        remote_ip, _ = conn.raddr

        if port_filters and local_port not in port_filters:
            continue

        if '%' in local_ip:
            local_ip = local_ip.split('%')[0]
        if remote_ip.startswith('::ffff:'):
            remote_ip = remote_ip.split('::ffff:')[1]

        key = (remote_ip, local_port)
        if not is_valid_ip(remote_ip) or key in seen:
            continue
        seen.add(key)

        info = get_ip_info(remote_ip)
        print(f"{remote_ip}:{local_port}  →  {info}")

if __name__ == "__main__":
    ports = set()
    for arg in sys.argv[1:]:
        try:
            ports.add(int(arg))
        except ValueError:
            sys.exit(f"Invalid port number: {arg}")

    port_filters = ports if ports else None
    show_connections(port_filters=port_filters)
